/*
 * Decompiled with CFR 0.152.
 */
package net.rasanovum.viaromana.map;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.minecraft.core.BlockPos;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.NbtAccounter;
import net.minecraft.nbt.NbtIo;
import net.minecraft.resources.ResourceKey;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.level.ChunkPos;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.storage.LevelResource;
import net.rasanovum.viaromana.CommonConfig;
import net.rasanovum.viaromana.ViaRomana;
import net.rasanovum.viaromana.map.MapBakeWorker;
import net.rasanovum.viaromana.map.MapInfo;
import net.rasanovum.viaromana.path.PathGraph;
import net.rasanovum.viaromana.surveyor.SurveyorUtil;

public final class ServerMapCache {
    private static final String MAP_DIR_NAME = "via_romana/network_images";
    private static final Map<UUID, MapInfo> cache = new ConcurrentHashMap<UUID, MapInfo>();
    private static final Map<UUID, Set<ChunkPos>> dirtyNetworks = new ConcurrentHashMap<UUID, Set<ChunkPos>>();
    private static final Map<ResourceKey<Level>, Set<ChunkPos>> scannedChunksByLevel = new ConcurrentHashMap<ResourceKey<Level>, Set<ChunkPos>>();
    private static final Set<UUID> modifiedForSaving = ConcurrentHashMap.newKeySet();
    private static ScheduledExecutorService scheduler;
    private static MinecraftServer minecraftServer;

    private ServerMapCache() {
    }

    public static void init(MinecraftServer server) {
        minecraftServer = server;
        ServerMapCache.shutdown();
        scheduler = Executors.newScheduledThreadPool(2);
        ViaRomana.LOGGER.info("Starting Via Romana schedulers...");
        scheduler.scheduleAtFixedRate(() -> ServerMapCache.runSafe(ServerMapCache::processAllDirtyNetworks, "Error during scheduled map reprocessing"), CommonConfig.map_refresh_interval, CommonConfig.map_refresh_interval, TimeUnit.SECONDS);
        ViaRomana.LOGGER.debug("Scheduled map reprocessing every {} seconds.", (Object)CommonConfig.map_refresh_interval);
        scheduler.scheduleAtFixedRate(() -> ServerMapCache.runSafe(() -> ServerMapCache.saveAllToDisk(false), "Error during scheduled map cache saving"), CommonConfig.map_save_interval, CommonConfig.map_save_interval, TimeUnit.MINUTES);
        ViaRomana.LOGGER.debug("Scheduled map saving every {} minutes.", (Object)CommonConfig.map_save_interval);
    }

    public static void shutdown() {
        if (scheduler != null && !scheduler.isShutdown()) {
            scheduler.shutdown();
            try {
                if (!scheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                    scheduler.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                scheduler.shutdownNow();
                Thread.currentThread().interrupt();
            }
            ViaRomana.LOGGER.info("Via Romana schedulers shut down.");
        }
    }

    public static void markChunkDirty(ServerLevel level, ChunkPos pos) {
        boolean isNewChunkThisCycle = scannedChunksByLevel.computeIfAbsent((ResourceKey<Level>)level.dimension(), k -> ConcurrentHashMap.newKeySet()).add(pos);
        if (!isNewChunkThisCycle) {
            return;
        }
        PathGraph graph = PathGraph.getInstance(level);
        if (graph == null) {
            return;
        }
        graph.findNetworksForChunk(pos).forEach(network -> {
            UUID networkId = network.id();
            dirtyNetworks.computeIfAbsent(networkId, k -> ConcurrentHashMap.newKeySet()).add(pos);
        });
    }

    public static void processAllDirtyNetworks() {
        if (!scannedChunksByLevel.isEmpty()) {
            ConcurrentHashMap<ResourceKey<Level>, Set<ChunkPos>> chunksToScan = new ConcurrentHashMap<ResourceKey<Level>, Set<ChunkPos>>(scannedChunksByLevel);
            scannedChunksByLevel.clear();
            chunksToScan.forEach((levelKey, chunks) -> {
                ServerLevel level = minecraftServer.getLevel(levelKey);
                if (level != null) {
                    minecraftServer.execute(() -> chunks.forEach(chunkPos -> SurveyorUtil.refreshChunkTerrain(level, chunkPos)));
                }
            });
        }
        if (dirtyNetworks.isEmpty()) {
            return;
        }
        ConcurrentHashMap<UUID, Set<ChunkPos>> toProcess = new ConcurrentHashMap<UUID, Set<ChunkPos>>(dirtyNetworks);
        dirtyNetworks.clear();
        ViaRomana.LOGGER.debug("Processing {} dirty networks in scheduled update batch.", (Object)toProcess.size());
        toProcess.forEach((networkId, chunks) -> {
            if (chunks != null && !chunks.isEmpty()) {
                minecraftServer.execute(() -> ServerMapCache.updateOrGenerateMapAsync(networkId, chunks));
            }
        });
    }

    private static CompletableFuture<MapInfo> updateOrGenerateMapAsync(UUID networkId, Collection<ChunkPos> chunksToUpdate) {
        return ServerMapCache.findLevelForNetwork(networkId).map(level -> {
            PathGraph.NetworkCache network;
            PathGraph graph = PathGraph.getInstance(level);
            PathGraph.NetworkCache networkCache = network = graph != null ? graph.getNetworkCache(networkId) : null;
            if (network == null) {
                ViaRomana.LOGGER.warn("Could not find network cache for network {}, aborting update.", (Object)networkId);
                return CompletableFuture.completedFuture(null);
            }
            return CompletableFuture.supplyAsync(() -> {
                MapInfo newResult;
                MapBakeWorker worker = new MapBakeWorker();
                MapInfo previousResult = cache.get(networkId);
                if (previousResult != null && previousResult.hasImageData()) {
                    ViaRomana.LOGGER.debug("Performing incremental update for network {}.", (Object)networkId);
                    newResult = worker.updateMap(previousResult, (Set<ChunkPos>)new HashSet<ChunkPos>(chunksToUpdate), (ServerLevel)level, network);
                } else {
                    ViaRomana.LOGGER.debug("Performing full bake for network {}.", (Object)networkId);
                    newResult = worker.bake(networkId, (ServerLevel)level, network.getMin(), network.getMax(), network.getNodesAsInfo());
                }
                cache.put(networkId, newResult);
                modifiedForSaving.add(networkId);
                ViaRomana.LOGGER.debug("Map update completed for network {}.", (Object)networkId);
                return newResult;
            }, (Executor)minecraftServer).exceptionally(ex -> {
                ViaRomana.LOGGER.error("Failed during map update for network {}", (Object)networkId, ex);
                return null;
            });
        }).orElseGet(() -> {
            ViaRomana.LOGGER.warn("Could not find level for network {}, aborting update.", (Object)networkId);
            return CompletableFuture.completedFuture(null);
        });
    }

    private static Optional<ServerLevel> findLevelForNetwork(UUID networkId) {
        for (ServerLevel level : minecraftServer.getAllLevels()) {
            PathGraph graph = PathGraph.getInstance(level);
            if (graph == null || graph.getNetworkCache(networkId) == null) continue;
            return Optional.of(level);
        }
        return Optional.empty();
    }

    public static Optional<MapInfo> getMapData(UUID networkId) {
        MapInfo result = cache.get(networkId);
        if (result != null) {
            return Optional.of(result);
        }
        return ServerMapCache.loadFromDisk(networkId);
    }

    public static CompletableFuture<MapInfo> generateMapIfNeeded(UUID networkId, ServerLevel level) {
        return ServerMapCache.getMapData(networkId).map(CompletableFuture::completedFuture).orElseGet(() -> {
            PathGraph.NetworkCache network;
            ViaRomana.LOGGER.debug("Generating initial map for network {} (client request).", (Object)networkId);
            PathGraph graph = PathGraph.getInstance(level);
            if (graph != null && (network = graph.getNetworkCache(networkId)) != null) {
                network.getNodesAsInfo().forEach(node -> SurveyorUtil.refreshChunkTerrain(level, new ChunkPos(node.position)));
                Set<ChunkPos> allChunks = network.getNodesAsInfo().stream().map(node -> new ChunkPos(node.position)).collect(Collectors.toSet());
                return ServerMapCache.updateOrGenerateMapAsync(networkId, allChunks);
            }
            ViaRomana.LOGGER.warn("Could not find network to generate map for networkId {}", (Object)networkId);
            return CompletableFuture.completedFuture(null);
        });
    }

    private static Optional<MapInfo> loadFromDisk(UUID networkId) {
        try {
            CompoundTag tag;
            byte[] png;
            Path mapDir = ServerMapCache.getMapDirectory();
            String base = "network-" + String.valueOf(networkId);
            Path pngPath = mapDir.resolve(base + ".png");
            Path metaPath = mapDir.resolve(base + ".nbt");
            try (InputStream pngStream = Files.newInputStream(pngPath, new OpenOption[0]);
                 InputStream metaStream = Files.newInputStream(metaPath, new OpenOption[0]);){
                png = pngStream.readAllBytes();
                tag = NbtIo.readCompressed((InputStream)metaStream, (NbtAccounter)new NbtAccounter(Long.MAX_VALUE, Integer.MAX_VALUE));
            }
            catch (NoSuchFileException e) {
                return Optional.empty();
            }
            BlockPos min = BlockPos.of((long)tag.getLong("min"));
            BlockPos max = BlockPos.of((long)tag.getLong("max"));
            int scale = tag.getInt("scale");
            MapInfo info = MapInfo.fromServerCache(networkId, min, max, List.of(), png, scale, List.of());
            cache.put(networkId, info);
            return Optional.of(info);
        }
        catch (IOException e) {
            ViaRomana.LOGGER.error("Failed to load map {} from disk", (Object)networkId, (Object)e);
            return Optional.empty();
        }
    }

    public static void invalidate(UUID networkId) {
        dirtyNetworks.remove(networkId);
        cache.remove(networkId);
        modifiedForSaving.remove(networkId);
        try {
            Path mapDir = ServerMapCache.getMapDirectory();
            String base = "network-" + String.valueOf(networkId);
            boolean pngDeleted = Files.deleteIfExists(mapDir.resolve(base + ".png"));
            boolean nbtDeleted = Files.deleteIfExists(mapDir.resolve(base + ".nbt"));
            if (pngDeleted || nbtDeleted) {
                ViaRomana.LOGGER.debug("Deleted map files from disk for network {}", (Object)networkId);
            }
        }
        catch (IOException e) {
            ViaRomana.LOGGER.error("Failed to delete map files from disk for network {}", (Object)networkId, (Object)e);
        }
    }

    public static void clear() {
        cache.clear();
        dirtyNetworks.clear();
        modifiedForSaving.clear();
        scannedChunksByLevel.clear();
    }

    public static void saveAllToDisk(boolean forceSave) {
        HashSet<UUID> networksToSave;
        if (modifiedForSaving.isEmpty() && !forceSave) {
            return;
        }
        HashSet<UUID> hashSet = networksToSave = forceSave ? new HashSet<UUID>(cache.keySet()) : new HashSet<UUID>(modifiedForSaving);
        if (networksToSave.isEmpty()) {
            return;
        }
        ViaRomana.LOGGER.debug("Saving {} maps to disk...", (Object)networksToSave.size());
        try {
            Path mapDir = ServerMapCache.getMapDirectory();
            Files.createDirectories(mapDir, new FileAttribute[0]);
            int savedCount = 0;
            for (UUID id : networksToSave) {
                MapInfo info = cache.get(id);
                if (info == null || !info.hasImageData()) {
                    ServerMapCache.invalidate(id);
                    continue;
                }
                String base = "network-" + String.valueOf(id);
                Path pngPath = mapDir.resolve(base + ".png");
                Path nbtPath = mapDir.resolve(base + ".nbt");
                CompoundTag tag = new CompoundTag();
                tag.putLong("min", info.minBounds().asLong());
                tag.putLong("max", info.maxBounds().asLong());
                tag.putInt("scale", info.bakeScaleFactor());
                if (info.createdAtMs() != null) {
                    tag.putLong("createdAt", info.createdAtMs().longValue());
                }
                try {
                    OutputStream pngOut = Files.newOutputStream(pngPath, new OpenOption[0]);
                    try {
                        OutputStream nbtOut = Files.newOutputStream(nbtPath, new OpenOption[0]);
                        try {
                            pngOut.write(info.pngData());
                            NbtIo.writeCompressed((CompoundTag)tag, (OutputStream)nbtOut);
                            ++savedCount;
                        }
                        finally {
                            if (nbtOut == null) continue;
                            nbtOut.close();
                        }
                    }
                    finally {
                        if (pngOut == null) continue;
                        pngOut.close();
                    }
                }
                catch (IOException e) {
                    ViaRomana.LOGGER.error("Failed to write map files for network {}", (Object)id, (Object)e);
                }
            }
            if (savedCount > 0) {
                ViaRomana.LOGGER.debug("Saved {} modified maps to disk.", (Object)savedCount);
            }
            modifiedForSaving.removeAll(networksToSave);
        }
        catch (IOException e) {
            ViaRomana.LOGGER.error("Failed to save map cache to disk.", (Throwable)e);
        }
    }

    public static void deleteAllMapsFromDisk() {
        block8: {
            ServerMapCache.clear();
            try {
                Path mapDir = ServerMapCache.getMapDirectory();
                if (!Files.exists(mapDir, new LinkOption[0])) break block8;
                try (Stream<Path> stream = Files.walk(mapDir, new FileVisitOption[0]);){
                    stream.sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete);
                }
                ViaRomana.LOGGER.debug("Deleted all map files from disk.");
            }
            catch (IOException e) {
                ViaRomana.LOGGER.error("Failed to delete map directory from disk.", (Throwable)e);
            }
        }
    }

    private static Path getMapDirectory() {
        return minecraftServer.getWorldPath(LevelResource.ROOT).resolve(MAP_DIR_NAME);
    }

    private static void runSafe(Runnable task, String errorMessage) {
        try {
            task.run();
        }
        catch (Exception e) {
            ViaRomana.LOGGER.error(errorMessage, (Throwable)e);
        }
    }
}

